/**
 * \file sdc_op_adv.h
 *
 * \brief LIBSDC: Definition of functions etc. of crytographic operations (advanced API) - <b>not recommended</b>
 *
 * Applications needing to perform wrap/unwrap, encrypt/decrypt or sign/verify
 * usually only need the functions provided by sdc_op_conv.h.\n
 * In case applications need more control on the individual steps of cryptographic
 * operations functions in this header file are required.\n
 * The common operation header or generic types like error codes are included internally.
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

/**
 * \ingroup operations_encdec
 * \defgroup operations_encdec_adv Advanced API
 * Advanced API
 */

/**
 * \ingroup operations_encdec_adv
 * \defgroup operations_encdec_adv_common Common
 * Common functions to e.g. estimate the overall message length
 */

/**
 * \ingroup operations_encdec_adv
 * \defgroup operations_encdec_adv_ext_form Extended formatted
 * Variant of \ref operations_encdec_conv_form providing full control on the parameters
 */

/**
 * \ingroup operations_encdec_adv
 * \defgroup operations_encdec_adv_iuf init, update, finalize
 * In contrast to \ref operations_encdec_conv_norm this API allows processing
 * the data in smaller slices using init, update and finalize functions.\n
 * This is crucial when processing huge data.\n
 * For example when processing a file of 42MB the \ref operations_encdec_conv_norm functions
 * would require about 84MB of memory (42MB to hold the input + 42MB for the output).\n
 * With init, update, finalize the same file could be processed in 42 slices of 1MB each.
 * That way the memory requirement can be reduced to about 2 times the size of each slice.\n
 * \n
 * Using init, update, finalize is always started by a single call to
 * init (\ref sdc_encrypt_init / \ref sdc_decrypt_init).\n
 * Afterwards the application can read a certain amount of data(e.g. from a file),
 * process this data using update (\ref sdc_encrypt_update / \ref sdc_decrypt_update) and
 * store the result(e.g. to a file).
 * The update step can be repeated as often as required to process the complete data.\n
 * After all input data is processed the application needs to call
 * finalize (\ref sdc_encrypt_finalize / \ref sdc_decrypt_finalize) to add or remove
 * padding and provide the last output bytes.\n
 * \n
 * Note: In case the data length is 0 it is allowed to not call update at all. But
 * calling init and finalize is mandatory.\n
 * \n
 * In contrast to \ref operations_encdec_conv_norm the output buffer is no longer
 * internally allocated but provided by the application to allow reusing the buffer
 * for all updates.\n
 * In order to compute the maximum output length for an update the functions
 * \ref sdc_encrypt_get_update_len and sdc_decrypt_get_update_len are provided.\n
 * Note: the actual output length of an update might be smaller than this length
 * depending on the internal state.\n
 * \n
 * Furthermore the maximal overall message length can be computed using
 * \ref sdc_encrypt_get_overall_len or \ref sdc_decrypt_get_overall_len.\n
 * \n
 * Beside the influence of memory consumption the selected slice size will
 * have an impact on the overall performance.\n
 * If the slice size is too small, more calls to process the data are required.
 * This can reduce the performance due to overhead of e.g. ioctls etc..\n
 * On the other side huge slice sizes will increase the required memory.
 * In case this length is bigger than what can be processed in a single call
 * to the hardware, it won't increase the performance.\n
 * In order to allow application to determine a reasonable size with good performance
 * and without unnecessary high memory requirements the function
 * \ref sdc_encrypt_decrypt_desc_get_max_chunk_len is provided.
 */

/**
 * \ingroup operations_signverify
 * \defgroup operations_signverify_adv Advanced API
 * Advanced API
 */

/**
 * \ingroup operations_signverify_adv
 * \defgroup operations_signverify_adv_common Common
 * Common functions to e.g. get max chunk length
 */

/**
 * \ingroup operations_signverify_adv
 * \defgroup operations_signverify_adv_ext_form Extended formatted
 * Variant of \ref operations_signverify_conv_form providing full control on the parameters
 */

/**
 * \ingroup operations_signverify_adv
 * \defgroup operations_signverify_adv_iuf init, update, finalize
 * In contrast to \ref operations_signverify_conv_norm this API allows processing
 * the data in smaller slices using init, update and finalize functions.\n
 * This is crucial when processing huge data.\n
 * For example when processing a file of 42MB the \ref operations_signverify_conv_norm functions
 * would require about 42MB + tag length of memory (42MB to hold the input tag length to hold output ).\n
 * With init, update, finalize the same file could be processed in 42 slices of 1MB each.
 * That way the memory requirement can be reduced to about 2 times the size of each slice.\n
 * \n
 * Using init, update, finalize is always started by a single call to
 * init (\ref sdc_sign_init / \ref sdc_verify_init).\n
 * Afterwards the application can read a certain amount of data(e.g. from a file),
 * process this data using update (\ref sdc_sign_update / \ref sdc_verify_update) and
 * store the result(e.g. to a file).
 * The update step can be repeated as often as required to process the complete data.\n
 * After all input data is processed the application needs to call
 * finalize (\ref sdc_sign_finalize / \ref sdc_verify_finalize) to get tag(mac or signature) output or to verify.\n
 * \n
 * Note: In case the data length is 0 it is allowed to not call update at all. But
 * calling init and finalize is mandatory.\n
 * \n
 * Output buffer is allocated internally in finalize call in sign operation to store the tag.
 * Beside the influence of memory consumption the selected slice size will
 * have an impact on the overall performance.\n
 * If the slice size is too small, more calls to process the data are required.
 * This can reduce the performance due to overhead of e.g. ioctls etc..\n
 * On the other side huge slice sizes will increase the required memory.
 * In case this length is bigger than what can be processed in a single call
 * to the hardware, it won't increase the performance.\n
 * In order to allow application to determine a reasonable size with good performance
 * and without unnecessary high memory requirements the function
 * \ref sdc_sign_verify_desc_get_max_chunk_len is provided.
 */

/**
 * \ingroup operations_wrapunwrap
 * \defgroup operations_wrapunwrap_adv Advanced API
 * Advanced API
 */

/**
  * \ingroup operations_wrapunwrap_adv
 * \defgroup operations_wrapunwrap_adv_common Common
 * Common functions to e.g. get max chunk length
 */

/**
 * \ingroup operations_wrapunwrap_adv
 * \defgroup operations_wrapunwrap_adv_ext_form Extended formatted
 * Variant of \ref operations_wrapunwrap_conv_form providing full control on the parameters
 */

/**
 * \ingroup operations_wrapunwrap_adv
 * \defgroup operations_wrapunwrap_adv_iuf init, update, finalize
 * In contrast to \ref operations_wrapunwrap_conv_norm this API allows processing
 * the data in smaller slices using init, update and finalize functions.\n
 * This is crucial when processing huge data.\n
 * For example when processing a file of 42MB the \ref operations_wrapunwrap_conv_norm functions
 * would require about 42MB(to hold input) + 42MB(to hold output) + tag length of memory.\n
 * With init, update, finalize the same file could be processed in 42 slices of 1MB each.
 * That way the memory requirement can be reduced to about 2 times the size of each slice.\n
 * \n
 * Using init, update, finalize is always started by a single call to
 * init (\ref sdc_wrap_init / \ref sdc_unwrap_init).\n
 * Afterwards the application can read a certain amount of aad data(e.g. from a file)
 * process this data using aad update (\ref sdc_wrap_aad_update / \ref sdc_unwrap_aad_update)\n
 * The aad update step can be repeated as often as required to set the complete aad data.\n
 * Note: No output produced in aad update.\n
 * Then application can read a certain amount of data(e.g. from a file),
 * process this data using update (\ref sdc_wrap_update / \ref sdc_unwrap_update) and
 * store the result(e.g. to a file).
 * The update step can be repeated as often as required to process the complete data.\n
 * After all input data is processed the application needs to call
 * finalize (\ref sdc_wrap_finalize / \ref sdc_unwrap_finalize) to provide the last output bytes and
 *  tag(mac or signature) or to verify.\n
 * \n
 * Note: In case the data length is 0 it is allowed to not call update at all. But
 * calling init and finalize is mandatory.\n
 * \n
 * In contrast to \ref operations_wrapunwrap_conv_norm the output buffer is no longer
 * internally allocated but provided by the application to allow reusing the buffer
 * for all updates.\n
 * In order to compute the maximum output length for an update the functions
 * \ref sdc_wrap_get_update_len and sdc_unwrap_get_update_len are provided.\n
 * Note: the actual output length of an update might be smaller than this length
 * depending on the internal state.\n
 * \n
 * Furthermore the maximal overall message length can be computed using
 * \ref sdc_wrap_get_overall_len or \ref sdc_unwrap_get_overall_len.\n
 * \n
 * tag buffer is allocated internally in finalize call in wrap operation to store the tag.
 * Beside the influence of memory consumption the selected slice size will
 * have an impact on the overall performance.\n
 * If the slice size is too small, more calls to process the data are required.
 * This can reduce the performance due to overhead of e.g. ioctls etc..\n
 * On the other side huge slice sizes will increase the required memory.
 * In case this length is bigger than what can be processed in a single call
 * to the hardware, it won't increase the performance.\n
 * In order to allow application to determine a reasonable size with good performance
 * and without unnecessary high memory requirements the function
 * \ref sdc_wrap_unwrap_desc_get_max_chunk_len is provided.
 */

/**
 * \ingroup operations_digest
 * \defgroup operations_digest_adv Advanced API
 * Advanced API
 */

/**
  * \ingroup operations_digest_adv
 * \defgroup operations_digest_adv_common Common
 * Common functions to e.g. get max chunk length
 */

/**
 * \ingroup operations_digest_adv
 * \defgroup operations_digest_adv_iuf init, update, finalize
 * In contrast to \ref operations_digest_conv_norm this API allows processing
 * the data in smaller slices using init, update and finalize functions.\n
 * This is crucial when processing huge data.\n
 * For example when processing a file of 42MB the \ref operations_digest_conv_norm functions
 * would require about 42MB + digest length of memory (42MB to hold the input, digest length to hold output).\n
 * With init, update, finalize the same file could be processed in 42 slices of 1MB each.
 * That way the memory requirement can be reduced to size of each slice.\n
 * \n
 * Using init, update, finalize is always started by a single call to init (\ref sdc_dgst_init).\n
 * Afterwards the application can read a certain amount of data(e.g. from a file),
 * process this data using update (\ref sdc_dgst_update).
 * The update step can be repeated as often as required to process the complete data.\n
 * Beside the influence of memory consumption the selected slice size will
 * have an impact on the overall performance.\n
 * If the slice size is too small, more calls to process the data are required.
 * This can reduce the performance due to overhead of e.g. ioctls etc..\n
 * On the other side huge slice sizes will increase the required memory.
 * In case this length is bigger than what can be processed in a single call
 * to the hardware, it won't increase the performance.\n
 * In order to allow application to determine a reasonable size with good performance
 * and without unnecessary high memory requirements the function
 * \ref sdc_dgst_desc_get_max_chunk_len is provided.
 * After all input data is processed the application needs to call
 * finalize (\ref sdc_dgst_finalize) to get the result of the digest computation.\n
 * \n
 * Note: In case the data length is 0 it is allowed to not call update at all. But
 * calling init and finalize is mandatory.\n
 * \n
 * Output(digest) buffer is allocated internally in finalize call to store the digest(e.g. to a file).
 */

#ifndef __SDC_LIB_OP_ADV_H_
#define __SDC_LIB_OP_ADV_H_

#ifdef __cplusplus
extern "C"
{
#endif

#include <sdc.h>
#include <sdc_session.h>
#include <sdc_op_common.h>

/* Definitions types and defaults */

/* Functions */

/**
 * \ingroup operations_wrapunwrap_adv_ext_form
 * \brief Perform wrap of a memory buffer and generate a formatted output buffer
 *
 * The used key for wrapping is the key currently assigned to the session.
 * This formatted output buffer contains\n
 * - key information,
 * - information on algorithm and block mode,
 * - iv
 * - wrapped data
 * - tag information (mac or signature)
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] in_data   pointer to the buffer containing the plain data to be wrapped (externally provided buffer)
 * \param[in] in_len    length of the \p in_data buffer (external specified length)
 * \param[in,out] iv    IV buffer\n
 *                      In case \p iv != NULL the external provided IV is used\n
 *                      In case \p iv == NULL the IV is generated internally
 * \param[in] iv_len    length of the IV\n
 *                      The length of the IV needs to be specified externally\n
 *                      In case \p iv_len == SDC_IV_USE_DEFAULT the architecture and \p type specific default is used.\n
 * \param[in] aad_data  pointer to buffer containing additional authenticated data (externally provided buffer)
 * \param[in] aad_len   length of \p aad_data \n
 *                      Currently not supported (has to be 0)\n
 *                      Reserved for future use
 *
 * \param[in] tag_len   length of the tag (mac or signature)\n
 *                      The length of the tag needs to be specified externally\n
 *                      In case \p tag_len == SDC_TAG_USE_DEFAULT the architecture and \p type specific default is used.\n
 * \param[out] formatted_data  pointer to return a pointer for the internally allocated buffer storing the formatted wrapped data.\n
 *                      \p formatted_data must not be NULL\n
 *                      Initially *\p formatted_data has to be NULL!\n
 *                      Application has to free this buffer after usage
 * \param[out] formatted_len   pointer to return the length of \p formatted_data \n
 *                      The length is determined internally - therefore the value of *\p formatted_len is ignored\n
 *                      \p formatted_len must not be NULL\n
 *
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IV_INVALID - IV or IV length is invalid
 * \return \ref SDC_TAG_DATA_INVALID - tag (mac or signature) length is invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_AAD_DATA_INVALID - aad data pointer or length is invalid
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_wrap_formatted_extended(sdc_session_t *session,
                                                const sdc_wrap_unwrap_type_t *type,
                                                const uint8_t *in_data, const size_t in_len,
                                                const uint8_t *iv, const size_t iv_len,
                                                const uint8_t *aad_data, const size_t aad_len,
                                                const size_t tag_len,
                                                uint8_t **formatted_data, size_t *formatted_len);


/**
 * \ingroup operations_encdec_adv_ext_form
 * \brief Perform encryption of a memory buffer and generate a formatted output buffer
 *
 * The used key for encryption is the key currently assigned to the session.
 * This formatted output buffer contains\n
 * - key information,
 * - information on algorithm and block mode,
 * - iv
 * - encrypted data
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] in_data   pointer to the buffer containing the plain data to be encrypted (externally provided buffer)
 * \param[in] in_len    length of the \p in_data buffer (external specified length)
 * \param[in,out] iv    IV buffer\n
 *                      In case \p iv != NULL the external provided IV is used\n
 *                      In case \p iv == NULL the IV is generated internally
 * \param[in] iv_len    length of the IV\n
 *                      The length of the IV needs to be specified externally\n
 *                      In case \p iv_len == SDC_IV_USE_DEFAULT the architecture and \p type specific default is used.\n
 * \param[out] formatted_data  pointer to return a pointer for the internally allocated buffer storing the formatted encrypted data.\n
 *                      \p formatted_data must not be NULL\n
 *                      Initially *\p formatted_data has to be NULL!\n
 *                      Application has to free this buffer after usage
 * \param[out] formatted_len   pointer to return the length of \p formatted_data \n
 *                      The length is determined internally - therefore the value of *\p formatted_len is ignored\n
 *                      \p formatted_len must not be NULL\n
 *
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IV_INVALID - IV or IV length is invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_encrypt_formatted_extended(sdc_session_t *session,
                                                   const sdc_encrypt_decrypt_type_t *type,
                                                   const uint8_t *in_data, const size_t in_len,
                                                   const uint8_t *iv, const size_t iv_len,
                                                   uint8_t **formatted_data, size_t *formatted_len);



/**
 * \ingroup operations_signverify_adv_ext_form
 * \brief Perform authentication of a memory buffer and generate a formatted output buffer
 *
 * The used key for authentication is the key currently assigned to the session.
 * This formatted output buffer contains\n
 * - key information,
 * - information on algorithm and hash used,
 * - iv
 * - plain data (authenticated by tag)
 * - tag information (mac or signature)
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] in_data   pointer to the buffer containing the plain data to be signed and stored in formatted_data (externally provided buffer)
 * \param[in] in_len    length of the \p in_data buffer (external specified length)
 * \param[in,out] iv    IV buffer\n
 *                      In case \p iv != NULL the external provided IV is used\n
 *                      In case \p iv == NULL the IV is generated internally
 * \param[in] iv_len    length of the IV\n
 *                      The length of the IV needs to be specified externally\n
 *                      In case \p iv_len == SDC_IV_USE_DEFAULT the architecture and \p type specific default is used.\n
 *
 * \param[in] tag_len   length of the tag (mac or signature)\n
 *                      The length of the tag needs to be specified externally\n
 *                      In case \p tag_len == SDC_TAG_USE_DEFAULT the architecture and \p type specific default is used.\n
 * \param[out] formatted_data  pointer to return a pointer for the internally allocated buffer storing the formatted authenticated data.\n
 *                      \p formatted_data must not be NULL\n
 *                      Initially *\p formatted_data has to be NULL!\n
 *                      Application has to free this buffer after usage
 * \param[out] formatted_len   pointer to return the length of \p formatted_data \n
 *                      The length is determined internally - therefore the value of *\p formatted_len is ignored\n
 *                      \p formatted_len must not be NULL\n
 *
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IV_INVALID - IV or IV length is invalid
 * \return \ref SDC_TAG_DATA_INVALID - tag (mac or signature) length is invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_sign_formatted_extended(sdc_session_t *session,
                                                const sdc_sign_verify_type_t *type,
                                                const uint8_t *in_data, const size_t in_len,
                                                const uint8_t *iv, const size_t iv_len,
                                                const size_t tag_len,
                                                uint8_t **formatted_data, size_t *formatted_len);


/**
 * \ingroup operations_encdec_adv_common
 * \brief Get max length an update call can handle by a single hardware call
 *
 * This is not the max length sdc_encrypt_update / sdc_decrypt_update can
 * process. It is only the max length which can be processed at once
 * by below architecture specific hardware/SW layers.
 * Longer length calls might be split into multiple smaller chunks.
 * Therefore this length a good indication for efficient init/update/finalize
 * processing.
 * Smaller length might not have best hardware efficiency (increased overhead).
 * Longer length usually only consume more memory without positive effect on the performance.
 *
 * \param[in] desc pointer to descriptor
 * \param[out] max_val if != NULL used to return the max value
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - desc pointer invalid
 */
SDC_API sdc_error_t sdc_encrypt_decrypt_desc_get_max_chunk_len(const sdc_encrypt_decrypt_desc_t *desc,
                                                               size_t *max_val);

/**
 * \ingroup operations_encdec_adv_common
 * \brief Compute max output length for encrypt
 *
 * Calculate the maximum output length when encrypting a complete message
 * of the given \p type.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[in] in_len    overall message length
 * \param[out] out_len_max upper limit of the output length
 *
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_encrypt_get_overall_len(sdc_session_t *session,
                                                const sdc_encrypt_decrypt_type_t *type,
                                                const sdc_encrypt_decrypt_desc_t *desc,
                                                const size_t in_len, size_t *out_len_max);

/**
 * \ingroup operations_encdec_adv_common
 * \brief Compute max output length for decrypt
 *
 * Calculate the maximum output length when decrypting a complete message
 * of the given \p type.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[in] in_len    length of the complete message
 * \param[out] out_len_max  upper limit of the output length
 *
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_decrypt_get_overall_len(sdc_session_t *session,
                                                const sdc_encrypt_decrypt_type_t *type,
                                                const sdc_encrypt_decrypt_desc_t *desc,
                                                const size_t in_len, size_t *out_len_max);

/**
 * \ingroup operations_encdec_adv_iuf
 * \brief Compute max output length for update or finaline (encrypt)
 *
 * Calculate the maximum output length of a single update or finalize
 * operation when encrypting a message of the given \p type.\n
 * This value is the upper limit for the given length \p in_len
 * and all smaller length.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[in] in_len    max length passed to any update call
 * \param[out] out_len_max  upper limit of the output length
 *
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_encrypt_get_update_len(sdc_session_t *session,
                                               const sdc_encrypt_decrypt_type_t *type,
                                               const sdc_encrypt_decrypt_desc_t *desc,
                                               const size_t in_len, size_t *out_len_max);

/**
 * \ingroup operations_encdec_adv_iuf
 * \brief Compute max output length for update or finaline (decrypt)
 *
 * Calculate the maximum output length of a single update or finalize
 * operation when decrypting a message of the given \p type.\n
 * This value is the upper limit for the given length \p in_len
 * and all smaller length.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[in] in_len    max length passed to any update call
 * \param[out] out_len_max  upper limit of the output length
 *
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_decrypt_get_update_len(sdc_session_t *session,
                                               const sdc_encrypt_decrypt_type_t *type,
                                               const sdc_encrypt_decrypt_desc_t *desc,
                                               const size_t in_len, size_t *out_len_max);

/**
 * \ingroup operations_encdec_adv_iuf
 * \brief Initialize init, update, finalize encryption sequence (\ref operations_encdec_adv_iuf)
 *
 * Initialize the session with given parameters.\n
 * Generate random IV if not specified explicitly(*\p iv != NULL) by application.
 *
 * Note: The overall sequence is described in \ref operations_encdec_adv_iuf
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[in,out] iv    pointer to the IV buffer pointer\n
 *                      IV can be provided externally (*\p iv != NULL) or\n
 *                      internally generated (*\p iv == NULL)\n
 *                      In both cases \p iv must not be NULL\n
 * \param[in,out] iv_len  pointer to the length of the IV\n
 *                      The length has to be specified externally\n
 *                      For the internally generated IV the architecture and \p type specific default can be selected by using
 *                      *\p iv_len == SDC_IV_USE_DEFAULT. In this case *\p iv_len will contain the actual length on success\n
 *                      \p iv_len must not be NULL
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IV_INVALID - iv data or length pointer is invalid or invalid length specified
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_encrypt_init(sdc_session_t *session,
                                     const sdc_encrypt_decrypt_type_t *type,
                                     const sdc_encrypt_decrypt_desc_t *desc,
                                     uint8_t **iv, size_t *iv_len);

/**
 * \ingroup operations_encdec_adv_iuf
 * \brief Initialize init, update, finalize decryption sequence (\ref operations_encdec_adv_iuf)
 *
 * Initialize the session with given parameters.\n
 *
 * Note: The overall sequence is described in \ref operations_encdec_adv_iuf
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[in] iv       pointer to the IV buffer (externally provided buffer)
 * \param[in] iv_len   length of the IV  (external specified length)
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IV_INVALID - iv data or length pointer is invalid or invalid length specified
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_decrypt_init(sdc_session_t *session,
                                     const sdc_encrypt_decrypt_type_t *type,
                                     const sdc_encrypt_decrypt_desc_t *desc,
                                     const uint8_t *iv, size_t iv_len);

/**
 * \ingroup operations_encdec_adv_iuf
 * \brief Process some data for encrypt (\ref operations_encdec_adv_iuf)
 *
 * This function can be called multiple times to encrypt a slice of the
 * input data.\n
 *
 * Note: The overall sequence is described in \ref operations_encdec_adv_iuf\n
 *
 * Internally the request is handled by one or multiple calls to the kernel.\n
 * In case of an error the buffer pointer and the length of output buffers are garbage.\n
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[in] in_data   pointer to the buffer containing the plain data to be encrypted (externally provided buffer)
 * \param[in] in_len    length of the \p in_data buffer (external specified length)
 * \param[out] out_data pointer to the output buffer.
 * \param[in,out] out_len   pointer to the output buffer length\n
 *                          input: available length / output: used length\n
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_encrypt_update(sdc_session_t *session,
                                       const sdc_encrypt_decrypt_type_t *type,
                                       const sdc_encrypt_decrypt_desc_t *desc,
                                       const uint8_t *in_data, const size_t in_len,
                                       uint8_t *out_data, size_t *out_len);

/**
 * \ingroup operations_encdec_adv_iuf
 * \brief Process some data for decrypt (\ref operations_encdec_adv_iuf)
 *
 * This function can be called multiple times to decrypt a slice of the
 * input data.\n
 *
 * Note: The overall sequence is described in \ref operations_encdec_adv_iuf\n
 *
 * Internally the request is handled by one or multiple calls to the kernel.\n
 * In case of an error the buffer pointer and the length of output buffers are garbage.\n
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[in] in_data   pointer to the buffer containing the plain data to be encrypted (externally provided buffer)
 * \param[in] in_len    length of the \p in_data buffer (external specified length)
 * \param[out] out_data pointer to the output buffer.
 * \param[in,out] out_len   pointer to the output buffer length\n
 *                          input: available length / output: used length\n
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_decrypt_update(sdc_session_t *session,
                                       const sdc_encrypt_decrypt_type_t *type,
                                       const sdc_encrypt_decrypt_desc_t *desc,
                                       const uint8_t *in_data, const size_t in_len,
                                       uint8_t *out_data, size_t *out_len);

/**
 * \ingroup operations_encdec_adv_iuf
 * \brief Finalize the encryt sequence (\ref operations_encdec_adv_iuf)
 *
 * This function is called after all data has been processed using
 * \ref sdc_encrypt_update to add padding and return potential internally
 * buffered data.\n
 * Calling this function after \ref sdc_encrypt_init is mandatory.
 *
 * Note: The overall sequence is described in \ref operations_encdec_adv_iuf\n
 *
 * Internally the request is handled by one or multiple calls to the kernel.\n
 * In case of an error the buffer pointer and the length of output buffers are garbage.\n
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[out] out_data pointer to the output buffer.
 * \param[in,out] out_len   pointer to the output buffer length\n
 *                          input: available length / output: used length\n
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - overall provided length is invalid
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_encrypt_finalize(sdc_session_t *session,
                                         const sdc_encrypt_decrypt_type_t *type,
                                         const sdc_encrypt_decrypt_desc_t *desc,
                                         uint8_t *out_data, size_t *out_len);

/**
 * \ingroup operations_encdec_adv_iuf
 * \brief Finalize the decryt sequence (\ref operations_encdec_adv_iuf)
 *
 * This function is called after all data has been processed using
 * \ref sdc_decrypt_update to return potential internally
 * buffered data and remove padding.\n
 * Calling this function after \ref sdc_decrypt_init is mandatory.
 *
 * Note: The overall sequence is described in \ref operations_encdec_adv_iuf\n
 *
 * Internally the request is handled by one or multiple calls to the kernel.\n
 * In case of an error the buffer pointer and the length of output buffers are garbage.\n
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[out] out_data pointer to the output buffer.
 * \param[in,out] out_len   pointer to the output buffer length\n
 *                          input: available length / output: used length\n
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - overall provided length is invalid
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_decrypt_finalize(sdc_session_t *session,
                                         const sdc_encrypt_decrypt_type_t *type,
                                         const sdc_encrypt_decrypt_desc_t *desc,
                                         uint8_t *out_data, size_t *out_len);

/**
 * \ingroup operations_signverify_adv_common
 * \brief Get max length an update call can handle by a single hardware call
 *
 * This is not the max length sdc_sign_update / sdc_verify_update can
 * process. It is only the max length which can be processed at once
 * by below architecture specific hardware/SW layers.
 * Longer length calls might be split into multiple smaller chunks.
 * Therefore this length a good indication for efficient init/update/finalize
 * processing.
 * Smaller length might not have best hardware efficiency (increased overhead).
 * Longer length usually only consume more memory without positive effect on the performance.
 *
 * \param[in] desc pointer to descriptor
 * \param[out] max_val if != NULL used to return the max value
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - desc pointer invalid
 */

SDC_API sdc_error_t sdc_sign_verify_desc_get_max_chunk_len(const sdc_sign_verify_desc_t *desc, size_t *max_val);

/**
 * \ingroup operations_signverify_adv_iuf
 * \brief Initialize init, update, finalize sign sequence (\ref operations_signverify_adv_iuf)
 *
 * Initialize the session with given parameters.\n
 * Generate random IV if not specified explicitly(*\p iv != NULL) by application.
 *
 * Note: The overall sequence is described in \ref operations_signverify_adv_iuf
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_sign_verify_desc_t descriptor
 * \param[in,out] iv    pointer to the IV buffer pointer\n
 *                      IV can be provided externally (*\p iv != NULL) or\n
 *                      internally generated (*\p iv == NULL)\n
 *                      In both cases \p iv must not be NULL\n
 * \param[in,out] iv_len  pointer to the length of the IV\n
 *                      The length has to be specified externally\n
 *                      For the internally generated IV the architecture and \p type specific default can be selected by using
 *                      *\p iv_len == SDC_IV_USE_DEFAULT. In this case *\p iv_len will contain the actual length on success\n
 *                      \p iv_len must not be NULL
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IV_INVALID - iv data or length pointer is invalid or invalid length specified
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */

SDC_API sdc_error_t sdc_sign_init(sdc_session_t *session,
                                  const sdc_sign_verify_type_t *type,
                                  const sdc_sign_verify_desc_t *desc,
                                  uint8_t **iv, size_t *iv_len);

/**
 * \ingroup operations_signverify_adv_iuf
 * \brief Initialize init, update, finalize verify sequence (\ref operations_signverify_adv_iuf)
 *
 * Initialize the session with given parameters.\n
 *
 * Note: The overall sequence is described in \ref operations_signverify_adv_iuf
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_sign_verify_desc_t descriptor
 * \param[in] iv       pointer to the IV buffer (externally provided buffer)
 * \param[in] iv_len   length of the IV  (external specified length)
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IV_INVALID - iv data or length pointer is invalid or invalid length specified
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_verify_init(sdc_session_t *session,
                                    const sdc_sign_verify_type_t *type,
                                    const sdc_sign_verify_desc_t *desc,
                                    const uint8_t *iv, size_t iv_len);

/**
 * \ingroup operations_signverify_adv_iuf
 * \brief Process some data for sign (\ref operations_signverify_adv_iuf)
 *
 * This function can be called multiple times to sign a slice of the
 * input data.\n
 *
 * Note: The overall sequence is described in \ref operations_signverify_adv_iuf\n
 *
 * Internally the request is handled by one or multiple calls to the kernel.\n
 * In case of an error return appropriate error code.\n
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_sign_verify_desc_t descriptor
 * \param[in] in_data   pointer to the buffer containing the plain data to be signed (externally provided buffer)
 * \param[in] in_len    length of the \p in_data buffer (external specified length)
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_sign_update(sdc_session_t *session,
                                    const sdc_sign_verify_type_t *type,
                                    const sdc_sign_verify_desc_t *desc,
                                    const uint8_t *in_data, const size_t in_len);

/**
 * \ingroup operations_signverify_adv_iuf
 * \brief Process some data for verify (\ref operations_signverify_adv_iuf)
 *
 * This function can be called multiple times to verify a slice of the
 * input data.\n
 *
 * Note: The overall sequence is described in \ref operations_signverify_adv_iuf\n
 *
 * Internally the request is handled by one or multiple calls to the kernel.\n
 * In case of an error return appropriate error code.\n
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_sign_verify_desc_t descriptor
 * \param[in] in_data   pointer to the buffer containing the plain data to be verified (externally provided buffer)
 * \param[in] in_len    length of the \p in_data buffer (external specified length)
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_verify_update(sdc_session_t *session,
                                      const sdc_sign_verify_type_t *type,
                                      const sdc_sign_verify_desc_t *desc,
                                      const uint8_t *in_data, const size_t in_len);

/**
 * \ingroup operations_signverify_adv_iuf
 * \brief Finalize the sign sequence (\ref operations_signverify_adv_iuf)
 *
 * This function is called after all data has been processed using
 * \ref sdc_sign_update to add padding and process potential internally
 * buffered data.\n
 * Calling this function after \ref sdc_sign_init is mandatory.
 *
 * Note: The overall sequence is described in \ref operations_signverify_adv_iuf\n
 *
 * In case of an error the tag is NULL and the length of tag is zero.\n
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_sign_verify_desc_t descriptor
 * \param[out] tag_data pointer to the output buffer(mac or signature).
 * \param[in,out] tag_len   pointer to the output buffer length\n
 *                          input: available length / output: used length\n
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - overall provided length is invalid
 * \return \ref SDC_TAG_DATA_INVALID - tag data or tag length pointer is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_sign_finalize(sdc_session_t *session,
                                      const sdc_sign_verify_type_t *type,
                                      const sdc_sign_verify_desc_t *desc,
                                      uint8_t **tag_data, size_t *tag_len);

/**
 * \ingroup operations_signverify_adv_iuf
 * \brief Finalize the verify sequence (\ref operations_signverify_adv_iuf)
 *
 * This function is called after all data has been processed using
 * \ref sdc_verify_update to add padding and process potential internally
 * buffered data.\n
 * Calling this function after \ref sdc_verify_init is mandatory.
 *
 * Note: The overall sequence is described in \ref operations_signverify_adv_iuf\n
 *
 * In case of an error the tag is NULL and the length of tag is zero.\n
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_sign_verify_desc_t descriptor
 * \param[out] tag_data pointer to the output buffer(mac or signature).
 * \param[in,out] tag_len   pointer to the output buffer length\n
 *                          input: available length / output: used length\n
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - overall provided length is invalid
 * \return \ref SDC_TAG_DATA_INVALID - tag data or tag length pointer is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_verify_finalize(sdc_session_t *session,
                                        const sdc_sign_verify_type_t *type,
                                        const sdc_sign_verify_desc_t *desc,
                                        const uint8_t *tag_data, size_t tag_len);

/**
 * \ingroup operations_digest_adv_common
 * \brief Get max length an update call can handle by a single hardware call
 *
 * This is not the max length sdc_dgst_update can process.
 * It is only the max length which can be processed at once
 * by below architecture specific hardware/SW layers.
 * Longer length calls might be split into multiple smaller chunks.
 * Therefore this length a good indication for efficient init/update/finalize
 * processing.
 * Smaller length might not have best hardware efficiency (increased overhead).
 * Longer length usually only consume more memory without positive effect on the performance.
 *
 * \param[in] desc pointer to descriptor
 * \param[out] max_val if != NULL used to return the max value
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - desc pointer invalid
 */
SDC_API sdc_error_t sdc_dgst_desc_get_max_chunk_len(const sdc_dgst_desc_t *desc, size_t *max_val);

/**
 * \ingroup operations_digest_adv_iuf
 * \brief Initialize init, update, finalize digest sequence (\ref operations_digest_adv_iuf)
 *
 * Initialize the session with given parameters.\n
 * Note: The overall sequence is described in \ref operations_digest_adv_iuf
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_dgst_desc_t descriptor
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_dgst_init(sdc_session_t *session,
                                  const sdc_dgst_type_t *type,
                                  const sdc_dgst_desc_t *desc);
/**
 * \ingroup operations_digest_adv_iuf
 * \brief Process some data for digest operation (\ref operations_digest_adv_iuf)
 *
 * This function can be called multiple times to digest a slice of the
 * input data.\n
 *
 * Note: The overall sequence is described in \ref operations_digest_adv_iuf\n
 *
 * Internally the request is handled by one or multiple calls to the kernel.\n
 * In case of an error return appropriate error code.\n
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_dgst_desc_t descriptor
 * \param[in] in_data   pointer to the buffer containing the plain data (externally provided buffer)
 * \param[in] in_len    length of the \p in_data buffer (external specified length)
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_dgst_update(sdc_session_t *session,
                                    const sdc_dgst_type_t *type,
                                    const sdc_dgst_desc_t *desc,
                                    const uint8_t *in_data, const size_t in_len);

/**
 * \ingroup operations_digest_adv_iuf
 * \brief Finalize the digest sequence (\ref operations_digest_adv_iuf)
 *
 * This function is called after all data has been processed using
 * \ref sdc_dgst_update to add padding and process potential internally
 * buffered data.\n
 * Calling this function after \ref sdc_dgst_init is mandatory.
 *
 * Note: digest_data needs to be initialized to NULL.
 * Note: The overall sequence is described in \ref operations_digest_adv_iuf\n
 *
 * In case of an error the digest is NULL and the length of digest is zero.\n
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_dgst_desc_t descriptor
 * \param[out] digest_data  pointer to the digest buffer.
 * \param[in,out] digest_len   pointer to the digest buffer length\n
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - overall provided length is invalid
 * \return \ref SDC_DGST_DATA_INVALID - digest or length pointer is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_dgst_finalize(sdc_session_t *session,
                                      const sdc_dgst_type_t *type,
                                      const sdc_dgst_desc_t *desc,
                                      uint8_t **digest_data, size_t *digest_len);

/**
 * \ingroup operations_wrapunwrap_adv_common
 * \brief Get max length an update call can handle by a single hardware call
 *
 * This is not the max length sdc_encrypt_update / sdc_decrypt_update can
 * process. It is only the max length which can be processed at once
 * by below architecture specific hardware/SW layers.
 * Longer length calls might be split into multiple smaller chunks.
 * Therefore this length a good indication for efficient init/update/finalize
 * processing.
 * Smaller length might not have best hardware efficiency (increased overhead).
 * Longer length usually only consume more memory without positive effect on the performance.
 *
 * \param[in] desc pointer to descriptor
 * \param[out] max_val if != NULL used to return the max value
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - desc pointer invalid
 */
SDC_API sdc_error_t sdc_wrap_unwrap_desc_get_max_chunk_len(const sdc_wrap_unwrap_desc_t *desc, size_t *max_val);

/**
 * \ingroup operations_wrapunwrap_adv_common
 * \brief Compute max output length for wrap
 *
 * Calculate the maximum output length when wrapping a complete message
 * of the given \p type.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_wrap_unwrap_desc_t descriptor
 * \param[in] in_len    overall message length
 * \param[out] out_len_max upper limit of the output length
 *
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_wrap_get_overall_len(sdc_session_t *session,
                                             const sdc_wrap_unwrap_type_t *type,
                                             const sdc_wrap_unwrap_desc_t *desc,
                                             const size_t in_len,
                                             size_t *out_len_max);

/**
 * \ingroup operations_wrapunwrap_adv_common
 * \brief Compute max output length for unwrap
 *
 * Calculate the maximum output length when unwrapping a complete message
 * of the given \p type.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_wrap_unwrap_desc_t descriptor
 * \param[in] in_len    length of the complete message
 * \param[out] out_len_max  upper limit of the output length
 *
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_unwrap_get_overall_len(sdc_session_t *session,
                                               const sdc_wrap_unwrap_type_t *type,
                                               const sdc_wrap_unwrap_desc_t *desc,
                                               const size_t in_len, size_t *out_len_max);

/**
 * \ingroup operations_wrapunwrap_adv_iuf
 * \brief Compute max output length for update or finalize (wrap)
 *
 * Calculate the maximum output length of a single update or finalize
 * operation when wrapping a message of the given \p type.\n
 * This value is the upper limit for the given length \p in_len
 * and all smaller length.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_wrap_unwrap_desc_t descriptor
 * \param[in] in_len    max length passed to any update call
 * \param[out] out_len_max  upper limit of the output length
 *
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */

SDC_API sdc_error_t sdc_wrap_get_update_len(sdc_session_t *session,
                                            const sdc_wrap_unwrap_type_t *type,
                                            const sdc_wrap_unwrap_desc_t *desc,
                                            const size_t in_len,
                                            size_t *out_len_max);

/**
 * \ingroup operations_wrapunwrap_adv_iuf
 * \brief Compute max output length for update or finalize (unwrap)
 *
 * Calculate the maximum output length of a single update or finalize
 * operation when unwrapping a message of the given \p type.\n
 * This value is the upper limit for the given length \p in_len
 * and all smaller length.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_wrap_unwrap_desc_t descriptor
 * \param[in] in_len    max length passed to any update call
 * \param[out] out_len_max  upper limit of the output length
 *
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_unwrap_get_update_len(sdc_session_t *session,
                                              const sdc_wrap_unwrap_type_t *type,
                                              const sdc_wrap_unwrap_desc_t *desc,
                                              const size_t in_len, size_t *out_len_max);

/**
 * \ingroup operations_wrapunwrap_adv_iuf
 * \brief Initialize init, update, finalize wrap sequence (\ref operations_wrapunwrap_adv_iuf)
 *
 * Initialize the session with given parameters.\n
 * Generate random IV if not specified explicitly(*\p iv != NULL) by application.
 *
 * Note: The overall sequence is described in \ref operations_wrapunwrap_adv_iuf
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_wrap_unwrap_desc_t descriptor
 * \param[in] total_inlen input data length which is going to be processed
 * \param[in] total_aadlen additional authenticated data length(external specified length)
 * \param[in,out] tag_len pointer to tag length\n
 *                      tag length can be provided explicitly (*\p tag_len != SDC_TAG_USE_DEFAULT ) or
 *                      get absolute tag length (*\p tag_len == SDC_TAG_USE_DEFAULT)\n
 *                      input: available length / output: absolute length\n
 * \param[in,out] iv    pointer to the IV buffer pointer\n
 *                      IV can be provided externally (*\p iv != NULL) or\n
 *                      internally generated (*\p iv == NULL)\n
 *                      In both cases \p iv must not be NULL\n
 * \param[in,out] iv_len  pointer to the length of the IV\n
 *                      The length has to be specified externally\n
 *                      For the internally generated IV the architecture and \p type specific default can be selected by using
 *                      *\p iv_len == SDC_IV_USE_DEFAULT.\n In this case *\p iv_len will contain the actual length on success
 *                      \p iv_len must not be NULL
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_TAG_DATA_INVALID - tag length is invalid
 * \return \ref SDC_AAD_DATA_INVALID -  additional authenticated data (aad) is invalid
 * \return \ref SDC_IV_INVALID - iv data or length pointer is invalid or invalid length specified
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_wrap_init(sdc_session_t *session,
                                  const sdc_wrap_unwrap_type_t *type,
                                  const sdc_wrap_unwrap_desc_t *desc,
                                  const size_t total_inlen,
                                  const size_t total_aadlen,
                                  size_t *tag_len,
                                  uint8_t **iv, size_t *iv_len);

/**
 * \ingroup operations_wrapunwrap_adv_iuf
 * \brief Initialize init, update, finalize unwrap sequence (\ref operations_wrapunwrap_adv_iuf)
 *
 * Initialize the session with given parameters.\n
 *
 * Note: The overall sequence is described in \ref operations_wrapunwrap_adv_iuf
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_wrap_unwrap_desc_t descriptor
 * \param[in] total_inlen total input data length which is going to be processed
 * \param[in] total_aadlen additional authenticated data length(external specified length)
 * \param[in] tag_len tag length(external specified length)
 * \param[in] iv       pointer to the IV buffer (externally provided buffer)
 * \param[in] iv_len   length of the IV  (external specified length)
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_TAG_DATA_INVALID - tag length is invalid
 * \return \ref SDC_AAD_DATA_INVALID -  additional authenticated data (aad) is invalid
 * \return \ref SDC_IV_INVALID - iv data or length pointer is invalid or invalid length specified
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_unwrap_init(sdc_session_t *session,
                                    const sdc_wrap_unwrap_type_t *type,
                                    const sdc_wrap_unwrap_desc_t *desc,
                                    const size_t total_inlen,
                                    const size_t total_aadlen,
                                    size_t tag_len,
                                    const uint8_t *iv, size_t iv_len);

/**
 * \ingroup operations_wrapunwrap_adv_iuf
 * \brief Process aad data for wrap (\ref operations_wrapunwrap_adv_iuf)
 *
 * This function can be called multiple times in slice to set aad data
 * for wrap operation.\n
 *
 * Note: The overall sequence is described in \ref operations_wrapunwrap_adv_iuf\n
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[in] aad_data  pointer to the buffer containing the aad data to be wrap (externally provided buffer)
 * \param[in] aad_len   length of the \p aad_len buffer (external specified length)
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_AAD_DATA_INVALID - aad data pointer or length is invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_wrap_aad_update(sdc_session_t *session,
                                        const sdc_wrap_unwrap_type_t *type,
                                        const sdc_wrap_unwrap_desc_t *desc,
                                        const uint8_t *aad_data, const size_t aad_len);

/**
 * \ingroup operations_wrapunwrap_adv_iuf
 * \brief Process aad data for unwrap (\ref operations_wrapunwrap_adv_iuf)
 *
 * This function can be called multiple times in slice to set aad data
 * for unwrap operation.\n
 *
 * Note: The overall sequence is described in \ref operations_wrapunwrap_adv_iuf\n
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[in] aad_data  pointer to the buffer containing the aad data to be unwrap (externally provided buffer)
 * \param[in] aad_len   length of the \p aad_len buffer (external specified length)
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_AAD_DATA_INVALID - aad data pointer or length is invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_unwrap_aad_update(sdc_session_t *session,
                                          const sdc_wrap_unwrap_type_t *type,
                                          const sdc_wrap_unwrap_desc_t *desc,
                                          const uint8_t *aad_data, const size_t aad_len);

/**
 * \ingroup operations_wrapunwrap_adv_iuf
 * \brief Process some data for wrap (\ref operations_wrapunwrap_adv_iuf)
 *
 * This function can be called multiple times to wrap a slice of the
 * input data.\n
 *
 * Note: The overall sequence is described in \ref operations_wrapunwrap_adv_iuf\n
 *
 * Internally the request is handled by one or multiple calls to the kernel.\n
 * In case of an error the buffer pointer and the length of output buffers are garbage.\n
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[in] in_data   pointer to the buffer containing the plain data to be wrapped (externally provided buffer)
 * \param[in] in_len    length of the \p in_data buffer (external specified length)
 * \param[out] out_data pointer to the output buffer.
 * \param[in,out] out_len   pointer to the output buffer length\n
 *                          input: available length / output: used length\n
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_wrap_update(sdc_session_t *session,
                                    const sdc_wrap_unwrap_type_t *type,
                                    const sdc_wrap_unwrap_desc_t *desc,
                                    const uint8_t *in_data, const size_t in_len,
                                    uint8_t *out_data, size_t *out_len);

/**
 * \ingroup operations_wrapunwrap_adv_iuf
 * \brief Process some data for unwrap (\ref operations_wrapunwrap_adv_iuf)
 *
 * This function can be called multiple times to unwrap a slice of the
 * input data.\n
 *
 * Note: The overall sequence is described in \ref operations_wrapunwrap_adv_iuf\n
 *
 * Internally the request is handled by one or multiple calls to the kernel.\n
 * In case of an error the buffer pointer and the length of output buffers are garbage.\n
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_encrypt_decrypt_desc_t descriptor
 * \param[in] in_data   pointer to the buffer containing the plain data to be encrypted (externally provided buffer)
 * \param[in] in_len    length of the \p in_data buffer (external specified length)
 * \param[out] out_data pointer to the output buffer.
 * \param[in,out] out_len   pointer to the output buffer length\n
 *                          input: available length / output: used length\n
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_unwrap_update(sdc_session_t *session,
                                      const sdc_wrap_unwrap_type_t *type,
                                      const sdc_wrap_unwrap_desc_t *desc,
                                      const uint8_t *in_data, const size_t in_len,
                                      uint8_t *out_data, size_t *out_len);

/**
 * \ingroup operations_wrapunwrap_adv_iuf
 * \brief Finalize the wrap sequence (\ref operations_wrapunwrap_adv_iuf)
 *
 * This function is called after all data has been processed using
 * \ref sdc_wrap_update to return potential internally buffered data.\n
 * Calling this function after \ref sdc_wrap_init is mandatory.
 *
 * Note: The overall sequence is described in \ref operations_wrapunwrap_adv_iuf\n
 *
 * In case of an error output buffer and output length, tag data buffer and tag data length are garbage.\n
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_wrap_unwrap_desc_t descriptor
 * \param[out] out_data pointer to the output buffer (externally provided buffer)
 * \param[in,out] out_len   pointer to the output buffer length\n
 *                          input: available length / output: used length\n
 * \param[out] tag_data  pointer to the tag buffer(mac or signature) (internally buffer allocated).
 * \param[in] tag_len       tag buffer length (externally specified length same value as returned by \ref sdc_wrap_init)
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - overall provided length is invalid
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return \ref SDC_TAG_DATA_INVALID - tag data or length is invalid.
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_wrap_finalize(sdc_session_t *session,
                                      const sdc_wrap_unwrap_type_t *type,
                                      const sdc_wrap_unwrap_desc_t *desc,
                                      uint8_t *out_data, size_t *out_len,
                                      uint8_t **tag_data, size_t tag_len);

/**
 * \ingroup operations_wrapunwrap_adv_iuf
 * \brief Finalize the unwrap sequence (\ref operations_wrapunwrap_adv_iuf)
 *
 * This function is called after all data has been processed using
 * \ref sdc_unwrap_update to return potential internally buffered data.\n
 * Calling this function after \ref sdc_unwrap_init is mandatory.
 *
 * Note: The overall sequence is described in \ref operations_wrapunwrap_adv_iuf\n
 *
 * Internally the request is handled by one or multiple calls to the kernel.\n
 * In case of an error output buffer and output length, tag data buffer and tag data length are garbage.\n
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] desc      filled \ref sdc_wrap_unwrap_desc_t descriptor
 * \param[out] out_data pointer to the output buffer.
 * \param[in,out] out_len   pointer to the output buffer length\n
 *                          input: available length / output: used length\n
 * \param[in] tag_data  pointer to the tag buffer(mac or signature) (externally buffer allocated).
 * \param[in] tag_len       tag buffer length (externally specified length same value as returned by \ref sdc_unwrap_init)
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - overall provided length is invalid
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return \ref SDC_TAG_DATA_INVALID - tag data or length is invalid.
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_unwrap_finalize(sdc_session_t *session,
                                        const sdc_wrap_unwrap_type_t *type,
                                        const sdc_wrap_unwrap_desc_t *desc,
                                        uint8_t *out_data, size_t *out_len,
                                        const uint8_t *tag_data, size_t tag_len);
#ifdef __cplusplus
}
#endif

#endif
